#region Using declarations
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.SuperDom;
using NinjaTrader.Gui.Tools;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.NinjaScript.DrawingTools;
#endregion

namespace NinjaTrader.NinjaScript.Indicators
{
    public class AdvancedTradingSignal : Indicator
    {
        private SMA fastSMA;
        private SMA slowSMA;
        private MACD macd;
        private RSI rsi;

        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        public int FastPeriod { get; set; } = 9;

        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        public int SlowPeriod { get; set; } = 21;

        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        public int RsiPeriod { get; set; } = 14;

        [Browsable(false)]
        [XmlIgnore()]
        public Series<double> BuySignalPlot { get; private set; }

        [Browsable(false)]
        [XmlIgnore()]
        public Series<double> SellSignalPlot { get; private set; }

        [Browsable(false)]
        [XmlIgnore()]
        public Series<double> MACDHistogramPlot { get; private set; }

        [Browsable(false)]
        [XmlIgnore()]
        public Series<double> RSIPlot { get; private set; }

        [Browsable(false)]
        [XmlIgnore()]
        public Series<double> TrendDotPlot { get; private set; }

        protected override void OnStateChange()
        {
            if (State == State.SetDefaults)
            {
                AddPlot(Brushes.Blue, "BuySignal");
                AddPlot(Brushes.Red, "SellSignal");
                AddPlot(new Stroke(Brushes.Green, DashStyleHelper.Solid, 5), PlotStyle.Bar, "MACDHistogram");
                AddPlot(Brushes.Orange, "RSI");
                AddLine(Brushes.LightGray, 50, "RSI_Midline");
                AddPlot(new Stroke(Brushes.DodgerBlue, DashStyleHelper.Solid, 2), PlotStyle.Dot, "TrendDot");
            }
            else if (State == State.DataLoaded)
            {
                fastSMA = SMA(FastPeriod);
                slowSMA = SMA(SlowPeriod);
                macd = MACD(12, 26, 9);
                rsi = RSI(RsiPeriod, 3);
                BuySignalPlot = new Series<double>(this);
                SellSignalPlot = new Series<double>(this);
                MACDHistogramPlot = new Series<double>(this);
                RSIPlot = new Series<double>(this);
                TrendDotPlot = new Series<double>(this);
            }
        }

        protected override void OnBarUpdate()
        {
            if (CurrentBar < Math.Max(FastPeriod, SlowPeriod))
                return;

            Values[2][0] = macd.Diff[0]; // MACD Histogram Visualization
            Values[3][0] = rsi[0]; // RSI Visualization
            Values[4][0] = 0; // Trend Dot at 0 line

            MACDHistogramPlot[0] = macd.Diff[0];
            RSIPlot[0] = rsi[0];
            TrendDotPlot[0] = 0;

            PlotBrushes[2][0] = macd.Diff[0] > 0 ? Brushes.Green : Brushes.Red;
            PlotBrushes[3][0] = (rsi[0] < 70 && rsi[1] < rsi[0] && rsi[2] < rsi[1]) ? Brushes.Green :
                                (rsi[0] > 30 && rsi[1] > rsi[0] && rsi[2] > rsi[1]) ? Brushes.Red : Brushes.Orange;
            PlotBrushes[4][0] = fastSMA[0] > slowSMA[0] ? Brushes.DodgerBlue : Brushes.HotPink;

            if (CrossAbove(fastSMA, slowSMA, 1) && macd.Diff[0] > 0 && rsi[0] < 70 && rsi[1] < rsi[0] && rsi[2] < rsi[1])
            {
                Values[0][0] = 1; // Buy signal
                BuySignalPlot[0] = 1;
                Draw.ArrowUp(this, "BuySignal" + CurrentBar, false, 0, Low[0] - TickSize, Brushes.Green);
                BackBrushes[0] = Brushes.LightGreen;
            }
            else
            {
                Values[0][0] = 0;
                BuySignalPlot[0] = 0;
            }

            if (CrossBelow(fastSMA, slowSMA, 1) && macd.Diff[0] < 0 && rsi[0] > 30 && rsi[1] > rsi[0] && rsi[2] > rsi[1])
            {
                Values[1][0] = -1; // Sell signal
                SellSignalPlot[0] = -1;
                Draw.ArrowDown(this, "SellSignal" + CurrentBar, false, 0, High[0] + TickSize, Brushes.Red);
                BackBrushes[0] = Brushes.LightCoral;
            }
            else
            {
                Values[1][0] = 0;
                SellSignalPlot[0] = 0;
            }
        }
    }
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private AdvancedTradingSignal[] cacheAdvancedTradingSignal;
		public AdvancedTradingSignal AdvancedTradingSignal(int fastPeriod, int slowPeriod, int rsiPeriod)
		{
			return AdvancedTradingSignal(Input, fastPeriod, slowPeriod, rsiPeriod);
		}

		public AdvancedTradingSignal AdvancedTradingSignal(ISeries<double> input, int fastPeriod, int slowPeriod, int rsiPeriod)
		{
			if (cacheAdvancedTradingSignal != null)
				for (int idx = 0; idx < cacheAdvancedTradingSignal.Length; idx++)
					if (cacheAdvancedTradingSignal[idx] != null && cacheAdvancedTradingSignal[idx].FastPeriod == fastPeriod && cacheAdvancedTradingSignal[idx].SlowPeriod == slowPeriod && cacheAdvancedTradingSignal[idx].RsiPeriod == rsiPeriod && cacheAdvancedTradingSignal[idx].EqualsInput(input))
						return cacheAdvancedTradingSignal[idx];
			return CacheIndicator<AdvancedTradingSignal>(new AdvancedTradingSignal(){ FastPeriod = fastPeriod, SlowPeriod = slowPeriod, RsiPeriod = rsiPeriod }, input, ref cacheAdvancedTradingSignal);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.AdvancedTradingSignal AdvancedTradingSignal(int fastPeriod, int slowPeriod, int rsiPeriod)
		{
			return indicator.AdvancedTradingSignal(Input, fastPeriod, slowPeriod, rsiPeriod);
		}

		public Indicators.AdvancedTradingSignal AdvancedTradingSignal(ISeries<double> input , int fastPeriod, int slowPeriod, int rsiPeriod)
		{
			return indicator.AdvancedTradingSignal(input, fastPeriod, slowPeriod, rsiPeriod);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.AdvancedTradingSignal AdvancedTradingSignal(int fastPeriod, int slowPeriod, int rsiPeriod)
		{
			return indicator.AdvancedTradingSignal(Input, fastPeriod, slowPeriod, rsiPeriod);
		}

		public Indicators.AdvancedTradingSignal AdvancedTradingSignal(ISeries<double> input , int fastPeriod, int slowPeriod, int rsiPeriod)
		{
			return indicator.AdvancedTradingSignal(input, fastPeriod, slowPeriod, rsiPeriod);
		}
	}
}

#endregion
