#region Using declarations
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.SuperDom;
using NinjaTrader.Gui.Tools;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.NinjaScript.DrawingTools;
#endregion

//This namespace holds Indicators in this folder and is required. Do not change it. 
namespace NinjaTrader.NinjaScript.Indicators.TradeSaber
{
	public class TSSwing : Indicator
	{
		private int				constant;
		private double			currentSwingHigh;
		private double			currentSwingLow;
		private ArrayList		lastHighCache;
		private double			lastSwingHighValue;
		private ArrayList		lastLowCache;
		private double			lastSwingLowValue;
		private int				saveCurrentBar;

		private Series<double>	swingHighSeries;
		private Series<double>	swingHighSwings;
		private Series<double>	swingLowSeries;
		private Series<double>	swingLowSwings;

		protected override void OnStateChange()
		{
			if (State == State.SetDefaults)
			{
				Description					= @"Swing indicator that only invalidates swings when price closes beyond the level.";
				Name						= "TSSwing";
				Calculate					= Calculate.OnBarClose;
				IsOverlay					= true;
				DisplayInDataBox			= false;
				PaintPriceMarkers			= false;
				IsSuspendedWhileInactive	= true;
				Strength					= 5;

				AddPlot(new Stroke(Brushes.DarkCyan, 2), PlotStyle.Dot, "SwingHigh");
				AddPlot(new Stroke(Brushes.Goldenrod, 2), PlotStyle.Dot, "SwingLow");
			}
			else if (State == State.Configure)
			{
				currentSwingHigh		= 0;
				currentSwingLow			= 0;
				lastSwingHighValue		= 0;
				lastSwingLowValue		= 0;
				saveCurrentBar			= -1;
				constant				= 2 * Strength + 1;
			}
			else if (State == State.DataLoaded)
			{
				lastHighCache	= new ArrayList();
				lastLowCache	= new ArrayList();

				swingHighSeries = new Series<double>(this);
				swingHighSwings = new Series<double>(this);
				swingLowSeries	= new Series<double>(this);
				swingLowSwings	= new Series<double>(this);
			}
		}

		protected override void OnBarUpdate()
		{
			double high0	= !(Input is PriceSeries || Input is Bars) ? Input[0] : High[0];
			double low0		= !(Input is PriceSeries || Input is Bars) ? Input[0] : Low[0];
			double close0	= !(Input is PriceSeries || Input is Bars) ? Input[0] : Close[0];

			// Handle tick replay / remove last bar scenarios
			if (BarsArray[0].BarsType.IsRemoveLastBarSupported && CurrentBar < saveCurrentBar)
			{
				currentSwingHigh			= SwingHighPlot.IsValidDataPoint(0) ? SwingHighPlot[0] : 0;
				currentSwingLow				= SwingLowPlot.IsValidDataPoint(0) ? SwingLowPlot[0] : 0;
				lastSwingHighValue			= swingHighSeries[0];
				lastSwingLowValue			= swingLowSeries[0];
				swingHighSeries[Strength]	= 0;
				swingLowSeries[Strength]	= 0;

				lastHighCache.Clear();
				lastLowCache.Clear();
				for (int barsBack = Math.Min(CurrentBar, constant) - 1; barsBack >= 0; barsBack--)
				{
					lastHighCache.Add(!(Input is PriceSeries or Data.Bars) ? Input[barsBack] : High[barsBack]);
					lastLowCache.Add(!(Input is PriceSeries or Data.Bars) ? Input[barsBack] : Low[barsBack]);
				}
				saveCurrentBar = CurrentBar;
				return;
			}

			if (saveCurrentBar != CurrentBar)
			{
				swingHighSwings[0]	= 0;
				swingLowSwings[0]	= 0;
				swingHighSeries[0]	= 0;
				swingLowSeries[0]	= 0;

				lastHighCache.Add(high0);
				if (lastHighCache.Count > constant)
					lastHighCache.RemoveAt(0);
				lastLowCache.Add(low0);
				if (lastLowCache.Count > constant)
					lastLowCache.RemoveAt(0);

				// Process Swing Highs
				if (lastHighCache.Count == constant)
				{
					bool	isSwingHigh				= true;
					double	swingHighCandidateValue	= (double)lastHighCache[Strength];
					
					for (int i = 0; i < Strength; i++)
						if (((double)lastHighCache[i]).ApproxCompare(swingHighCandidateValue) >= 0)
							isSwingHigh = false;

					for (int i = Strength + 1; i < lastHighCache.Count; i++)
						if (((double)lastHighCache[i]).ApproxCompare(swingHighCandidateValue) > 0)
							isSwingHigh = false;

					swingHighSwings[Strength] = isSwingHigh ? swingHighCandidateValue : 0.0;
					if (isSwingHigh)
						lastSwingHighValue = swingHighCandidateValue;

					if (isSwingHigh)
					{
						// New swing high detected - this replaces the old one
						currentSwingHigh = swingHighCandidateValue;
						for (int i = 0; i <= Strength; i++)
							SwingHighPlot[i] = currentSwingHigh;
					}
					else if (close0 > currentSwingHigh && currentSwingHigh.ApproxCompare(0.0) != 0)
					{
						// CLOSE is above swing high - invalidate it but still plot on this bar
						SwingHighPlot[0] = currentSwingHigh;
						currentSwingHigh = 0.0;
					}
					else if (currentSwingHigh.ApproxCompare(0.0) == 0)
					{
						// No valid swing high
						SwingHighPlot.Reset();
					}
					else
					{
						// Swing high still valid (price may have crossed intrabar but didn't close above)
						SwingHighPlot[0] = currentSwingHigh;
					}

					if (isSwingHigh)
						for (int i = 0; i <= Strength; i++)
							swingHighSeries[i] = lastSwingHighValue;
					else
						swingHighSeries[0] = lastSwingHighValue;
				}

				// Process Swing Lows
				if (lastLowCache.Count == constant)
				{
					bool	isSwingLow				= true;
					double	swingLowCandidateValue	= (double)lastLowCache[Strength];
					
					for (int i = 0; i < Strength; i++)
						if (((double)lastLowCache[i]).ApproxCompare(swingLowCandidateValue) <= 0)
							isSwingLow = false;

					for (int i = Strength + 1; i < lastLowCache.Count; i++)
						if (((double)lastLowCache[i]).ApproxCompare(swingLowCandidateValue) < 0)
							isSwingLow = false;

					swingLowSwings[Strength] = isSwingLow ? swingLowCandidateValue : 0.0;
					if (isSwingLow)
						lastSwingLowValue = swingLowCandidateValue;

					if (isSwingLow)
					{
						// New swing low detected - this replaces the old one
						currentSwingLow = swingLowCandidateValue;
						for (int i = 0; i <= Strength; i++)
							SwingLowPlot[i] = currentSwingLow;
					}
					else if (close0 < currentSwingLow && currentSwingLow.ApproxCompare(0.0) != 0 && currentSwingLow.ApproxCompare(double.MaxValue) != 0)
					{
						// CLOSE is below swing low - invalidate it but still plot on this bar
						SwingLowPlot[0] = currentSwingLow;
						currentSwingLow = double.MaxValue;
					}
					else if (currentSwingLow.ApproxCompare(0.0) == 0 || currentSwingLow.ApproxCompare(double.MaxValue) == 0)
					{
						// No valid swing low
						SwingLowPlot.Reset();
					}
					else
					{
						// Swing low still valid (price may have crossed intrabar but didn't close below)
						SwingLowPlot[0] = currentSwingLow;
					}

					if (isSwingLow)
						for (int i = 0; i <= Strength; i++)
							swingLowSeries[i] = lastSwingLowValue;
					else
						swingLowSeries[0] = lastSwingLowValue;
				}

				saveCurrentBar = CurrentBar;
			}
			else if (CurrentBar >= constant - 1)
			{
				// Intrabar updates - just keep plotting current values, don't invalidate on intrabar crosses
				if (lastHighCache.Count == constant && high0.ApproxCompare((double)lastHighCache[lastHighCache.Count - 1]) > 0)
					lastHighCache[lastHighCache.Count - 1] = high0;
				if (lastLowCache.Count == constant && low0.ApproxCompare((double)lastLowCache[lastLowCache.Count - 1]) < 0)
					lastLowCache[lastLowCache.Count - 1] = low0;

				// Keep plotting swing high if valid (regardless of intrabar price action)
				if (currentSwingHigh.ApproxCompare(0.0) != 0)
					SwingHighPlot[0] = currentSwingHigh;

				// Keep plotting swing low if valid (regardless of intrabar price action)
				if (currentSwingLow.ApproxCompare(double.MaxValue) != 0 && currentSwingLow.ApproxCompare(0.0) != 0)
					SwingLowPlot[0] = currentSwingLow;
			}
		}

		#region Functions
		/// <summary>
		/// Returns the number of bars ago a swing low occurred. Returns -1 if not found within lookBackPeriod.
		/// </summary>
		public int SwingLowBar(int barsAgo, int instance, int lookBackPeriod)
		{
			if (instance < 1)
				throw new Exception(string.Format("TSSwing.SwingLowBar: instance must be >= 1, was {0}", instance));
			if (barsAgo < 0)
				throw new Exception(string.Format("TSSwing.SwingLowBar: barsAgo must be >= 0, was {0}", barsAgo));
			if (barsAgo >= Count)
				throw new Exception(string.Format("TSSwing.SwingLowBar: barsAgo {0} >= Count {1}", barsAgo, Count));

			Update();

			for (int idx = CurrentBar - barsAgo - Strength; idx >= CurrentBar - barsAgo - Strength - lookBackPeriod; idx--)
			{
				if (idx < 0)
					return -1;
				if (idx >= swingLowSwings.Count)
					continue;

				if (swingLowSwings.GetValueAt(idx).Equals(0.0))
					continue;

				if (instance == 1)
					return CurrentBar - idx;

				instance--;
			}

			return -1;
		}

		/// <summary>
		/// Returns the number of bars ago a swing high occurred. Returns -1 if not found within lookBackPeriod.
		/// </summary>
		public int SwingHighBar(int barsAgo, int instance, int lookBackPeriod)
		{
			if (instance < 1)
				throw new Exception(string.Format("TSSwing.SwingHighBar: instance must be >= 1, was {0}", instance));
			if (barsAgo < 0)
				throw new Exception(string.Format("TSSwing.SwingHighBar: barsAgo must be >= 0, was {0}", barsAgo));
			if (barsAgo >= Count)
				throw new Exception(string.Format("TSSwing.SwingHighBar: barsAgo {0} >= Count {1}", barsAgo, Count));

			Update();

			for (int idx = CurrentBar - barsAgo - Strength; idx >= CurrentBar - barsAgo - Strength - lookBackPeriod; idx--)
			{
				if (idx < 0)
					return -1;
				if (idx >= swingHighSwings.Count)
					continue;

				if (swingHighSwings.GetValueAt(idx).Equals(0.0))
					continue;

				if (instance <= 1)
					return CurrentBar - idx;

				instance--;
			}

			return -1;
		}
		#endregion

		#region Properties
		[Range(1, int.MaxValue), NinjaScriptProperty]
		[Display(Name = "Strength", GroupName = "Parameters", Order = 0)]
		public int Strength { get; set; }

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> SwingHigh
		{
			get
			{
				Update();
				return swingHighSeries;
			}
		}

		private Series<double> SwingHighPlot
		{
			get
			{
				Update();
				return Values[0];
			}
		}

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> SwingLow
		{
			get
			{
				Update();
				return swingLowSeries;
			}
		}

		private Series<double> SwingLowPlot
		{
			get
			{
				Update();
				return Values[1];
			}
		}
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private TradeSaber.TSSwing[] cacheTSSwing;
		public TradeSaber.TSSwing TSSwing(int strength)
		{
			return TSSwing(Input, strength);
		}

		public TradeSaber.TSSwing TSSwing(ISeries<double> input, int strength)
		{
			if (cacheTSSwing != null)
				for (int idx = 0; idx < cacheTSSwing.Length; idx++)
					if (cacheTSSwing[idx] != null && cacheTSSwing[idx].Strength == strength && cacheTSSwing[idx].EqualsInput(input))
						return cacheTSSwing[idx];
			return CacheIndicator<TradeSaber.TSSwing>(new TradeSaber.TSSwing(){ Strength = strength }, input, ref cacheTSSwing);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.TradeSaber.TSSwing TSSwing(int strength)
		{
			return indicator.TSSwing(Input, strength);
		}

		public Indicators.TradeSaber.TSSwing TSSwing(ISeries<double> input , int strength)
		{
			return indicator.TSSwing(input, strength);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.TradeSaber.TSSwing TSSwing(int strength)
		{
			return indicator.TSSwing(Input, strength);
		}

		public Indicators.TradeSaber.TSSwing TSSwing(ISeries<double> input , int strength)
		{
			return indicator.TSSwing(input, strength);
		}
	}
}

#endregion
